# Hua - A PowerShell Static Content Generator

# $Author: jeffr $
# $Date: 2025-11-28 09:21:53 -0500 (Fri, 28 Nov 2025) $
# $Revision: 302 $

# Show version info and quit
if ($args[0] -eq '-V' -or $args[0] -eq '--version') {
	Write-Host 'Hua version: $Revision: 302 $'
	exit 0
}

# Usage information
$usage = "Usage: pwsh hua.ps1 [path to configuration file] [optional: -O|--orphans, -A 'ID'|--article 'ID' -G|--groff]"

#
# Get file and config variables
#

# Config file passed as first argument
if ($args[0]) {
	# Make sure the config file exists
	if (Test-Path $args[0] -PathType Leaf) {
		# Config file exists
		$configs = Get-Content -Path $args[0]
		foreach ($config in $configs) {
			if ($config -match '^*=') {
				# Set config variables
				$var_val = $config.split('=')
				$var_val = $var_val.replace(' ', '')
				New-Variable -Name $var_val[0] -Value $var_val[1]
			}
		}
	} else {
		# Config file doesn't exist
		Write-Host 'Configuration file error. File path specified as' $args[0]'. Quitting.'
		exit 2
	}
} else {
	# No config file specified
	Write-Host $usage
	exit 1
}

# Check to make sure all file/config vars are defined or quit
$config_vars = 'entries_file', 'meta_file', 'content_dir', 'markdown_dir', 'output_dir', 'index_file', 'archive_file', 'header_file', 'footer_file', 'read_more_file', 'comments_file', 'blog_root', 'web_root', 'title_sep', 'entries_pp', 'next_label', 'prev_label', 'log_dir', 'log_file'
foreach ($config_var in $config_vars) {
	if (-Not (Test-Path variable:local:$config_var)) {
		Write-Host "File/configuration variable error. $config_var variable is not defined. Quitting."
		exit 3
	}
}

# Check for existence of log file dir and begin logging (or quit)
if (Test-Path -Path $log_dir) {
	$log_path = Join-Path -Path $log_dir -ChildPath $log_file
	$run_date = Get-Date
	$cfg_file = $args[0]
	Add-Content -Path $log_path -Value ""
	Add-Content -Path $log_path -Value "Hua started $run_date using configuration file $cfg_file."
	Start-Transcript $log_path -Append | Out-Null
} else {
	Write-Host "Log file directory ($log_dir) does not exist. Quitting."
	exit 4
}

#
# ProcessTags
#
# 	input:	an array of tags
#	output: an html list of unique sorted tags
#
function ProcessTags {
	$tags = $args[0] | Sort-Object -Unique
	foreach ($tag in $tags) {
		$tag_link = $tag.replace(' ', '-')
		$tags_list += "<li><a href=`"tagged-with-$tag_link.html`"><span class=`"tag`">$tag</span></a></li>"
	}
	$tags_list
}

#
# ListOrphans
#
# 	input:	contents of entries_file, an array of output_dir article filenames
#	output: a list of orphans to stdout
#
function ListOrphans {
	foreach ($out_path in $args[1]) {
		$out_file = Split-Path $out_path -leaf
		$out_file_no_ext = [System.IO.Path]::GetFileNameWithoutExtension($out_file) + '.md'

		# If the output file is not represented in the entries file, it may be an orphan
		if (-Not ($args[0].'File'.contains($out_file)) -And (-Not ($args[0].'File'.contains($out_file_no_ext))) -And (-Not ($out_file -match 'tagged-with-*'))) {
			Write-Host $out_file
		}
	}
}

#
# MasterTags
#
# 	input: a file handle, tag list string, scope (1 or all)
#	output: addition/removal of master tag list in output HTML
#
function MasterTags {
	$fh = $args[0]
	$tl = $args[1]
	$sc = $args[2]

	if (Test-Path $fh -PathType Leaf) {
		$fh_content = Get-Content -Raw $fh
	} else {
		continue
	}

	[regex]$pattern = "--master_tags--"

	# If no --master_tags--, processing has already occurred on file so skip
	if (-Not ($fh_content -match $pattern)) {
		continue
	}

	if ($sc -eq 1) {
		# Replace the first
		$fh_content = $pattern.replace($fh_content, '<ul class="tags">' + $tl + '</ul>', 1)
	} else {
		# Replace all
		$fh_content = $pattern.replace($fh_content, '<ul class="tags">' + $tl + '</ul>')
	}

	# Write out the new file
	Set-Content $fh $fh_content
}

#
# IndexFileLup
#
# 	input: Entry ID
#	output: Filename of index page on which it appears
#
function IndexFileLup {
	$id = $args[0]
	$entries_l = Import-Csv -Path 'paging.csv' -Delimiter ","
	$page = $entries_l | Where-Object {$_.'ID' -eq $id} | ForEach-Object Page
	return $page
}

#
# TaggedWithLup
#
# 	input: Entry ID and tag
#	output: Filename of tagged with page on which it appears
#
function TaggedWithLup {
	$id = $args[0]
	$tag = $args[1]
	$entries_l = Import-Csv -Path 'paging.csv' -Delimiter ","

	$tags_l = $entries_l | Where-Object {$_.'ID' -eq $id} | ForEach-Object 'Tags'
	$tw_l = $entries_l | Where-Object {$_.'ID' -eq $id} | ForEach-Object 'TW Page'

	$tags_l = $tags_l -split '[|]'
	$tw_l = $tw_l -split '[|]'

	# Get position of tag
	$tag_i = $tags_l.IndexOf($tag)
	return $tw_l[$tag_i]
}

#
# PagingUi
#
# 	input: Page filename
#	output: Links to previous and next pages
#
function PagingUi {
	# Return nothing if index is single page
	$entries_l = Import-Csv -Path 'paging.csv' -Delimiter ","
	$page_f = 0
	foreach ($entry_t in $entries_l) {
		if ($entry_t.'Page' -like "*-2*") {
			$page_f = 1
		}
	}
	if ($page_f -eq 0) {
		return
	}
	
	# Get page number
	$index_base = [io.path]::GetFileNameWithoutExtension($page_old)
	$index_file_base = [io.path]::GetFileNameWithoutExtension($index_file)
	$pn = $index_base.replace("$index_file_base-", '')

	if ($pn -eq $index_base) {
		# First page
		$next_link = '<a href="' + $index_file_base + '-2.html">' + $next_label + '</a>'
		return '<p class="hua-paging">' + $next_link + '</p>'
	} else {
		# Page 2 and on
		if ($page -ne $page_old) {
			$pn_p = [int]$pn - 1
			$pn_n = [int]$pn + 1
			# Need exception for previous link to first page
			if ($pn_p -eq 1) {
				$prev_link = '<a href="' + "$index_file_base.html" + '">' + $prev_label + '</a>'
				$next_link = '<a href="' + $index_file_base + '-' + $pn_n + '.html">' + $next_label + '</a>'
				return '<p class="hua-paging">' + "$prev_link $next_link" + '</p>'
			} else {
				$prev_link = '<a href="' + $index_file_base + '-' + $pn_p + '.html">' + $prev_label + '</a>'
				$next_link = '<a href="' + $index_file_base + '-' + $pn_n + '.html">' + $next_label + '</a>'
				return '<p class="hua-paging">' + "$prev_link $next_link" + '</p>'
			}
		} else {
			$pn_p = [int]$pn - 1
			$prev_link = '<p class="hua-paging"><a href="' + $index_file_base + '-' + $pn_p + '.html">' + $prev_label + '</a></p>'
			return $prev_link
		}
	}
}

#
# CleanUpExit
#
# 	input: Exit status
#	output: File cleanup, stop transcript, and exit
#
function CleanUpExit {
	if (-Not ($null -eq $output_dir_tmp)) {
		if (Test-Path -Path $output_dir_tmp) {
			Remove-Item $output_dir_tmp -Recurse -Force
		}
	}
	if (Test-Path -Path 'paging.csv' -PathType Leaf) {
		Remove-Item 'paging.csv'
	}
	Stop-Transcript | Out-Null
	exit $args[0]
}

# Orphans or article mode passed as second argument
$orphans_mode = 0
$article_mode = 0
$groff_mode   = 0
if ($args[1]) {
	if ($args[1] -eq '-O' -or $args[1] -eq '--orphans') {
		$orphans_mode = 1
	} elseif ($args[1] -eq '-A' -or $args[1] -eq '--article') {
		$article_mode = 1
		$article_id = $args[2]
		# Groff mode?
		if ($args[3] -eq '-G' -or $args[3] -eq '--groff') {
			$groff_mode = 1
		}
	} else {
		Write-Host $usage
		CleanUpExit 13
	}
}

# Open the entries file and read its contents (or fail)
if (Test-Path $entries_file -PathType Leaf) {
	$entries = Import-Csv -Path $entries_file -Delimiter "," | Sort-Object { [int]($_.ID -replace '\D') } -Descending
} else {
	Write-Host "Entries file error. Entries file path set as $entries_file. Quitting."
	CleanUpExit 5
}

#
# Check for possible ID, file, and filename conflicts (e.g., entry.html and entry.md)
#

# IDs
$ids = @()
foreach ($entry in $entries) {
	$ids += $entry.ID
}
$ids_u = $ids | Sort-Object | Get-Unique

# Quit if the entries file doesn't contain any articles
if ($ids.Length -lt 2) {
	Write-Host 'No articles in entries file. Quitting.'
	CleanUpExit 6
}

if ($ids.Length -ne $ids_u.Length) {
	Write-Host 'Entry ID conflict. Check the entries file for repeated IDs. Quitting.'
	CleanUpExit 7
}

$conflict_f = 0
# Entries file
$fns = @()
foreach ($entry in $entries) {
	if ($entry.'ID' -eq '000000') { continue }
	$fn = [System.IO.Path]::GetFileNameWithoutExtension($entry.File)
	$fns += $fn
}

$fns_u = @($fns | Sort-Object | Get-Unique)
if ($fns.Length -ne $fns_u.Length) { $conflict_f = 1 }

# Content directory exists?
if (-Not (Test-Path $content_dir -PathType Container)) {
	Write-Host "Content directory error. Content directory path set as $content_dir. Quitting."
	CleanUpExit 15
}

# Content directory
$fns = @()
$fns_fp = Get-ChildItem -File $content_dir
foreach ($fn in $fns_fp) {
	$fn = $fn.Basename
	$fns += $fn
}
$fns_u = @($fns | Sort-Object | Get-Unique)
if ($fns.Length -ne $fns_u.Length) { $conflict_f = 1 }

if ($conflict_f -ne 0) {
	Write-Host 'Filename conflict. Check the entries file and content directory. Quitting.'
	Write-Host
	Write-Host 'NOTE: This can happen when an HTML file and a Markdown file have the same basename.'
	Write-Host '      For example: entry.html and entry.md conflict.'
	CleanUpExit 8
}

# Paging
& "$PSScriptRoot/hua-paging.ps1" $entries_file $entries_pp $index_file $content_dir

# Open the meta info file and read its contents (or fail)
if (Test-Path $meta_file -PathType Leaf) {
	$metas = Import-Csv -Path $meta_file -Delimiter "|"
} else {
	Write-Host "Meta info file error. Meta file path set as $meta_file. Quitting."
	CleanUpExit 9
}

# Orphans mode? Then get a list of files currently in the output directory
if ($orphans_mode -eq 1) {
	if (Test-Path -Path $output_dir -PathType Container) {
		$output_articles = Get-Item(Join-Path -Path $output_dir -ChildPath '*')
	} else {
		Write-Host "Output directory error. Output directory path set as $output_dir. Quitting."
		CleanUpExit 16
	}

	# Add archive and index files to entries so they're not counted as orphans
	$entries += [PSCustomObject]@{ File=[System.IO.Path]::GetFileName($archive_file) }

	# Account for all index files
	$entries_l = Import-Csv -Path 'paging.csv' -Delimiter ","
	foreach ($entry_l in $entries_l) {
		$entries += [PSCustomObject]@{ File=[System.IO.Path]::GetFileName($entry_l.'Page') }
	}

	ListOrphans $entries $output_articles
	CleanUpExit 0
}

# Make sure content and output directories differ
if ($output_dir -eq $content_dir) {
	Write-Host "Output ($output_dir) and content ($content_dir) directories must differ. Quitting."
	CleanUpExit 17
}

# Make temporary output directory and index/archive files
$output_dir_tmp = (Join-Path -Path $output_dir "temp")

# Make sure output directory is a directory
if (Test-Path -Path $output_dir -PathType container) {
	New-Item -ItemType Directory -Path $output_dir_tmp | Out-Null
} else {
	Write-Host "Output directory error. Output directory path set as $output_dir. Quitting."
	CleanUpExit 12
}

if ($article_mode -ne 1) {
	$index_file_tmp = (Join-Path -Path $output_dir_tmp (Split-Path $index_file -leaf))
	$archive_file_tmp = (Join-Path -Path $output_dir_tmp (Split-Path $archive_file -leaf))
}

# Get the header content (or fail)
if (Test-Path $header_file -PathType Leaf) {
	$header = Get-Content -Path $header_file
	# Account for empty include
	if ($null -eq $header) {
		Write-Host "Header file error. File ($header_file) is empty. Quitting."
		CleanUpExit 10
	}
} else {
	Write-Host "Header file error. Path set as $header_file. Quitting."
	CleanUpExit 10
}

# Get the footer content (or fail)
if (Test-Path $footer_file -PathType Leaf) {
	$footer = Get-Content -Path $footer_file
	# Account for empty include
	if ($null -eq $footer) {
		Write-Host "Footer file error. File ($footer_file) is empty. Quitting."
		CleanUpExit 11
	}
} else {
	Write-Host "Footer file error. Path set as $footer_file. Quitting."
	CleanUpExit 11
}

# Get content of read more link include (or error)
if (Test-Path $read_more_file -PathType Leaf) {
	$read_more = Get-Content -Path $read_more_file
	# Account for empty include
	if ($null -eq $read_more) {
		Write-Host "Read more file error. File ($read_more_file) is empty. Continuing."
		$read_more = ''
	} 
} else {
	Write-Host "Read more file error. Path set as $read_more_file. Continuing."
	$read_more = ''
}

# Get content of comments include (or error)
if (Test-Path $comments_file -PathType Leaf) {
	$comments = Get-Content -Path $comments_file
	# Account for empty include
	if ($null -eq $comments) { 
		Write-Host "Comments file error. File ($comments_file) is empty. Continuing."
		$comments = ''
	} 
} else {
	Write-Host "Comments file error. Path set as $comments_file. Continuing."
	$comments = ''
}

#
# Process Markdown (check to see if MarkdownToHtml is installed and template exists)
#
$md_exists = Test-Path (Join-Path -Path $markdown_dir -ChildPath 'md-template.html') -PathType Leaf
$no_md = 0

# Article mode?
if ($article_mode -eq 1) {
	# Get file to process based upon ID passed
	$article_file = $entries | Where-Object {$_.'ID' -eq $article_id} | ForEach-Object 'File'
} else {
	$article_file = '*.md'
}

# Process markdown if markdown module installed and template exists
if ((Get-Module -ListAvailable -Name 'MarkdownToHTML') -And ($md_exists)) {
	if ($article_file -eq '*.md') {
		# Process all markdown files
		foreach ($entry in $entries) {
			# Skip if file doesn't exist
			if (-Not (Test-Path (Join-Path -Path $content_dir -ChildPath $entry.'File'))) { continue }

			if ([System.IO.Path]::GetExtension($entry.'File') -eq '.md') {
				Convert-MarkdownToHTML -Verbose:$false -Path (Join-Path -Path $content_dir -ChildPath $entry.'File') -SiteDirectory $content_dir -Template $markdown_dir | Out-Null
			}
		}
	} else {
		# Article mode. Process individual markdown file
		
		# Check to see if file exists and is a markdown file
		if ((Test-Path (Join-Path -Path $content_dir -ChildPath $article_file)) -And [System.IO.Path]::GetExtension($article_file) -eq '.md') {
			Convert-MarkdownToHTML -Verbose:$false -Path (Join-Path -Path $content_dir -ChildPath $article_file) -SiteDirectory $content_dir -Template $markdown_dir | Out-Null
		}			
	}
} else {
	# Flag that markdown module isn't installed or template doesn't exist
	$no_md = 1
}

#
# Loop over the entries (skip the first) and output the content
#

# Array for the all tags list
$master_tag_list = @()

# Default meta info
$def_keys = $metas | Where-Object {$_.ID -eq '000000'} | ForEach-Object Keywords
$def_desc = $metas | Where-Object {$_.ID -eq '000000'} | ForEach-Object Description
$def_auth = $entries | Where-Object {$_.ID -eq '000000'} | ForEach-Object Author

# Write the index and archive file headers
$header = $header.replace('--meta_keywords--', $def_keys)
$header = $header.replace('--meta_description--', $def_desc)
$header = $header.replace('--meta_author--', $def_auth)

if ($article_mode -ne 1) {
	Add-Content $archive_file_tmp ($header -replace '--article_title--', "$title_sep Archive")
	Add-Content $archive_file_tmp "<h2>Available Articles</h2><ol id=`"archive`">"
}

$page_old = ''
$article_found = 0

foreach ($entry in $entries) {
	# Skip default entry
	if ($entry.'ID' -eq '000000') {
		continue
	}

	# Move on if we're in article mode and this isn't the target article
	if ($article_mode -eq 1 -And $entry.'ID' -ne $article_id) {
		continue
	} else {
		$article_found = 1
	}

	# No entry ID? Warn and skip
	if (-Not ($entry.'ID')) {
		Write-Host 'Entries file error. There is an entry without an ID. Skipping.'
		continue
	}

	$e_id = $entry.'ID'
	
	# No filename? Warn and skip
	if (-Not ($entry.'File')) {
		# Write-Host 'Entries file error. There is an entry without a filename. Skipping.'
		Write-Host "Entries file error. There is an entry (ID: $e_id) without a filename. Skipping."
		continue
	}

	# Content file missing? Warn and skip
	if (-Not (Test-Path (Join-Path -Path $content_dir -ChildPath $entry.'File') -PathType Leaf)) {
		$content_fp = (Join-Path -Path $content_dir -ChildPath $entry.'File')
		Write-Host "Entry content error. Entry content file path set as $content_fp. Skipping."
		continue
	}

	# Handle markdown filenames
	if ([System.IO.Path]::GetExtension($entry.File) -eq '.md') {
		$e_fn = [System.IO.Path]::GetFileNameWithoutExtension($entry.File) + '.html'
	} else {
		$e_fn = $entry.'File'
	}

	# Skip markdown files if no MarkdownToHTML module installed
	if ([System.IO.Path]::GetExtension($entry.File) -eq '.md' -and $no_md -eq 1) {
		Write-Host 'MarkdownToHTML module not installed or markdown template not found. Skipping.'
		continue
	}

	if ($entry.'Title') { $e_title = $entry.'Title' } else { $e_title = 'Untitled' }    # Article title
	if ($entry.'Tags') { $e_tags = $entry.'Tags'.split('|') } else { $e_tags = @() }    # Article tags
	if ($entry.'Date') { $e_date = $entry.'Date' } else { $e_date = '' }                # Article date
	if ($entry.'Author') { $e_author = $entry.'Author' } else { $e_author = '' }        # Article author
	if ($entry.'Contact') { $e_contact = $entry.'Contact' } else { $e_contact = '' }    # Article contact/email

	# Array for the article tag list
	$article_tag_list = @()

	# Loop over the tags
	foreach ($tag in $e_tags) {
		$article_tag_list += $tag.Trim()
		$master_tag_list += $tag.Trim()
	}

	# 
	# Handle related articles
	# 
	if ($entry.'Related') {
		$related_c = '<ul>'
		$e_related = $entry.'Related'.split('|')
		foreach ($related_a in $e_related) {
			# Is the ID valid?
			$test_a = $entries | Where-Object {$_.ID -eq $related_a}
			if ($test_a) {
				$related_f = $test_a.'File'.replace('.md', '.html')
				$related_c = $related_c + '<li><a href="' + $related_f + '">' + $test_a.'Title' + '</a></li>'
			} else {
				Write-Host "Related article error in $e_id. Article ID $related_a does not exist. Ignoring."
			}
		}
		$related_c = $related_c + '</ul>'
	} else {
		$related_c = ''
	}

	#
	# Create/append to the index/permalink file content
	#

	# Get the article content in content_dir
	$e_content = Get-Content -Path (Join-Path -Path $content_dir -ChildPath $e_fn)
	
	# If article file is empty, skip and continue
	if ($e_content.length -eq 0) {
		Write-Host "Content file error. File (ID: $e_id) is empty. Skipping."
		continue
	}

	# Handle paging for index file(s)
	if ($article_mode -ne 1) {
		$page = IndexFileLup($e_id)
		if ($page -ne $page_old) {
			# Footer for previous page (if not first page)
			if ($page_old -ne '') {
				$footer_ui = PagingUi($page_old)
				$footer_i = $footer.replace('<!-- paging -->', $footer_ui)
				Add-Content $index_file_tmp $footer_i
			}
			# Header for new page
			$index_file_tmp = (Join-Path -Path $output_dir_tmp (Split-Path $page -leaf))
			Add-Content $index_file_tmp ($header -replace '--article_title--', '')
		}
	}

	# Article permalink, date, title, author, etc.
	$e_content = $e_content.replace('--permalink_url--', $e_fn)
	$e_content = $e_content.replace('--article_date--', $e_date)
	$e_content = $e_content.replace('--article_date--', $e_date)
	$e_content = $e_content.replace('--article_title--', $e_title)
	$e_content = $e_content.replace('--author--', $e_author)
	$e_content = $e_content.replace('--contact--', $e_contact)
	$e_content = $e_content.replace('--related_articles--', $related_c)

	# Article mode groff output (via Pandoc)
	if ($groff_mode -eq 1) {
		# Check to see if Pandoc is in path
		if (Get-Command "pandoc" -ErrorAction SilentlyContinue) {
			# Remove tag placeholders from HTML
			$e_content_groff = $e_content.replace('--article_tags--', '')
			$e_content_groff = $e_content_groff.replace('--master_tags--', '')

			# Get entry file name without extension
			$e_fn_noext = [io.path]::GetFileNameWithoutExtension($e_fn)

			# Generate groff (ms)
			$groff_tmp = Join-Path -Path $output_dir_tmp -ChildPath "$e_fn_noext.tmp"
			$groff_out = Join-Path -Path $output_dir -ChildPath "$e_fn_noext.ms"
			Add-Content $groff_tmp $e_content_groff
			& "pandoc" -f html -t ms $groff_tmp -o $groff_out
			Remove-Item $groff_tmp

			# On Unix? Try to produce PDF
			if ([environment]::OSVersion.Platform -eq 'Unix') {
				if (Get-Command "groff" -ErrorAction SilentlyContinue) {
					# Produce postscript
					$ps_out =  Join-Path -Path $output_dir_tmp -ChildPath "$e_fn_noext.ps"
					& "groff" -Tps -ms $groff_out 1> $ps_out 2>/dev/null

					# Produce PDF
					if (Get-Command "ps2pdf" -ErrorAction SilentlyContinue) {
						$pdf_out =  Join-Path -Path $output_dir_tmp -ChildPath "$e_fn_noext.pdf"
						& "ps2pdf" $ps_out $pdf_out
					} else {
						Write-Host 'ps2pdf does not exist in path. PDF not generated.'
					}
				} else {
					Write-Host 'groff does not exist in path. Postscript not generated.'
				}	
			}
		} else {
			Write-Host "Pandoc does not exist in path. groff not generated."
		}
	}

	# Article-specific tags
	$e_tags_p = ProcessTags $e_tags
	$e_content = $e_content.replace('--article_tags--', '<ul class="tags">' + $e_tags_p + '</ul>')

	if ($article_mode -ne 1) {
		Add-Content $index_file_tmp $e_content
		# Permalink (read more link)
		Add-Content $index_file_tmp $read_more.replace('--link--', $e_fn)
	}
	
	# Create and add an entry to the archive file
	if ($article_mode -ne 1) {
		$archive_content = "<li><cite><a href=`"$e_fn`">$e_title</a></cite>, by $e_author (written $e_date).</li>"
		Add-Content $archive_file_tmp $archive_content
	}

	#
	# Create the permalink file in the temporary output directory
	#

	$permalink_path = (Join-Path -Path $output_dir_tmp -ChildPath $e_fn)

	# "Reset" header content so title can be properly substituted
	$header_tmp = Get-Content -Path $header_file
	$header_tmp = $header_tmp.replace('--article_title--', $title_sep + ' ' + $e_title)

	# Meta info lookup
	$keywords = $metas | Where-Object {$_.ID -eq $e_id} | ForEach-Object Keywords
	$description = $metas | Where-Object {$_.ID -eq $e_id} | ForEach-Object Description
	if ($null -eq $keywords -or $keywords -eq '') {
		$header_tmp = $header_tmp.replace('--meta_keywords--', $def_keys)
	} else {
		$header_tmp = $header_tmp.replace('--meta_keywords--', $keywords)
	}

	if ($null -eq $description) {
		$header_tmp = $header_tmp.replace('--meta_description--', $def_desc)
	} else {
		$header_tmp = $header_tmp.replace('--meta_description--', $description)
	}
	$header_tmp = $header_tmp.replace('--meta_author--', $e_author)

	Add-Content $permalink_path $header_tmp
	Add-Content $permalink_path $e_content

	# Substitute values in comments include
	$e_comments = $comments.replace('--this.page.url--', "$web_root$blog_root$e_fn")
	$e_comments = $e_comments.replace('--this.page.identifier--', $e_id)

	Add-Content $permalink_path $e_comments
	Add-Content $permalink_path $footer

	# 
	# Tag specific files
	#

	# For each article tag, add the article content to the "tagged-with" file
	if ($article_mode -ne 1) {
		foreach ($article_tag in $article_tag_list) {
			$tagged_with_fn = TaggedWithLup $e_id $article_tag
			
			# $tagged_with_fn = "tagged-with-$article_tag_url.html"
			$tagged_with_path = (Join-Path -Path $output_dir_tmp -ChildPath $tagged_with_fn)
			
			# If the tagged-with file is new, add the header first
			if (-Not (Test-Path $tagged_with_path -PathType Leaf)) {
				# "Reset" header content so title can be properly substituted
				$header_tmp = Get-Content -Path $header_file
				$header_tmp = $header_tmp.replace('--article_title--',  "$title_sep Articles tagged with &quot;$article_tag&quot;")

				# Meta tags
				$header_tmp = $header_tmp.replace('--meta_keywords--', $article_tag)
				$header_tmp = $header_tmp.replace('--meta_description--', "Articles tagged with $article_tag.")
				$header_tmp = $header_tmp.replace('--meta_author--', $def_auth)

				Add-Content $tagged_with_path $header_tmp
			}
			
			# Write the article content to the file
			Add-Content $tagged_with_path $e_content
			Add-Content $tagged_with_path $read_more.replace('--link--', $e_fn)
		}
	}

	$page_old = $page
}

# Warning for article mode and article not found
if ($article_mode -eq 1 -And $article_found -eq 0) {
	Write-Host 'The specified ID does not exist in the entries file.'
	CleanUpExit 14
}

# Paging for last page
$footer_ui = PagingUi($page_old)
$footer_i = $footer.replace('<!-- paging -->', $footer_ui)

if ($article_mode -ne 1) {
	Add-Content $index_file_tmp $footer_i
	Add-Content $archive_file_tmp "</ol>$footer"
}

#
# Post-process "tagged-with" files
#

$tw_pre_l = @()
foreach ($tagged_with_file in Get-Item(Join-Path -Path $output_dir_tmp -ChildPath 'tagged-with-*')) {
	# Single page or multiple pages?
	if ($tagged_with_file -match '-[*0-9].html$') {
		# Multiple, get prefix of filename
		$tagged_with_pre = $tagged_with_file -replace '-[*0-9].html$', ''
		$tw_pre_l = $tw_pre_l + $tagged_with_pre
	}
}

$tw_pre_l = $tw_pre_l | Select-Object -Unique
foreach ($tw_pre_l_i in $tw_pre_l) {
	$tw_pre_base = Split-Path -Path $tw_pre_l_i -Leaf
	$tw_files = Get-ChildItem -Path "$output_dir_tmp\$tw_pre_base*" | Sort-Object { [int]($_ -replace '\D') }
	$tw_c = 0
	$tw_files_l = $tw_files.Length # No. of pages

	foreach ($tw_file in $tw_files) {
		$tw_c ++ # Increment page counter
		if ($tw_c -eq 1) {
			# Condition for first page
			$prev_link = ''
			$next_link = '<a href="' + $tw_pre_base + '-2.html">' + $next_label + '</a>'
		} elseif ($tw_c -lt $tw_files_l) {
			# Condition for middle page
			$prev_pn = $tw_c - 1
			$next_pn = $tw_c + 1
			$prev_link = '<a href="' + $tw_pre_base + '-' + $prev_pn + '.html">' + $prev_label + '</a>'
			if ($prev_pn -eq 1) {
				$prev_link = $prev_link.replace('-1', '')
			}
			$next_link = '<a href="' + $tw_pre_base + '-' + $next_pn + '.html">' + $next_label + '</a>'
		} else {
			# Condition for last page
			$prev_pn = $tw_c - 1
			$prev_link = '<a href="' + $tw_pre_base + '-' + $prev_pn + '.html">' + $prev_label + '</a>'
			if ($prev_pn -eq 1) {
				$prev_link = $prev_link.replace('-1', '')
			}
			$next_link = ''
		}
		
		$page_links = '<p class="hua-paging">' + $prev_link + ' ' + $next_link + '</p>'
		$footer_i = $footer.replace('<!-- paging -->', $page_links)
		Add-Content $tw_file $footer_i
	}
}

# Account for footer in non-multi-page tagged-with files
foreach ($tagged_with_file in Get-Item(Join-Path -Path $output_dir_tmp -ChildPath 'tagged-with-*')) {
	if (-Not (Select-String -Path $tagged_with_file -pattern 'hua-paging')) {
		$footer_i = $footer.replace('<!-- paging -->', '')
		Add-Content $tagged_with_file $footer_i
	}
}

#
# Post-processing for the master tag list
#

$master_tag_list = ProcessTags $master_tag_list

# Permalink files
foreach ($entry in $entries) {

	# Handle markdown filenames
	if ([System.IO.Path]::GetExtension($entry.File) -eq '.md') {
		
		# Skip if no MarkdownToHTML installed
		if ($no_md -eq 1) { continue }
		$e_fn = [System.IO.Path]::GetFileNameWithoutExtension($entry.File) + '.html'
	} else {
		$e_fn = $entry.'File'
	}

	MasterTags (Join-Path -Path $output_dir_tmp -ChildPath $e_fn) $master_tag_list
}

# Tagged-with files
foreach ($tagged_with_file in Get-Item(Join-Path -Path $output_dir_tmp -ChildPath 'tagged-with-*')) {
	MasterTags $tagged_with_file $master_tag_list 1
}

foreach ($tagged_with_file in Get-Item(Join-Path -Path $output_dir_tmp -ChildPath 'tagged-with-*')) {
	MasterTags $tagged_with_file ''
}

# Handle the index page(s)
$index_base = Split-Path ([io.path]::GetFileNameWithoutExtension($index_file)) -leaf
$index_pages = Get-ChildItem (Join-Path -Path $output_dir_tmp -ChildPath "$index_base*")
foreach ($index_page in $index_pages) {
	MasterTags $index_page $master_tag_list 1
	MasterTags $index_page ''
}

#
# Compare files generated in temp with those in output directory and replace
#

foreach ($tmp_file in Get-Item(Join-Path -Path $output_dir_tmp -ChildPath '*')) {
	$destination_file = (Join-Path $output_dir -ChildPath (Split-Path $tmp_file -leaf))

	# If the file exists in the output directory, compare
	if (Test-Path (Join-Path $output_dir -ChildPath (Split-Path $tmp_file -leaf))) {
		# Will compare
		if(Compare-Object -ReferenceObject $(Get-Content -Path $destination_file) -DifferenceObject $(Get-Content -Path $tmp_file)) {
			# Replace tagged-with-* file
			Move-Item -Path $tmp_file -Destination $destination_file -Force
		}
	} else {
		# New tagged-with file. Simply move it
		Move-Item -Path $tmp_file -Destination $destination_file -Force
	}
}

#
# Clean up files generated from markdown
#

if ($no_md -ne 1) {
	foreach ($entry in $entries) {

		if ([System.IO.Path]::GetExtension($entry.File) -eq '.md') {
			$e_fn = [System.IO.Path]::GetFileNameWithoutExtension($entry.File) + '.html'

			if (Test-Path (Join-Path -Path $content_dir -ChildPath $e_fn) -PathType Leaf) {
				Remove-Item (Join-Path -Path $content_dir -ChildPath $e_fn)
			} else {
				continue
			}
		}
	}
}

CleanUpExit 0